
{ *********************************************************************** }
{                                                                         }
{ Delphi Visual Component Library                                         }
{                                                                         }
{ Copyright (c) 2004-2005 Borland Software Corporation                    }
{                                                                         }
{ *********************************************************************** }

unit Borland.Vcl.Win32;

interface

uses
  System.Runtime.InteropServices,
  System.Threading,
  System.Reflection,
  System.Reflection.Emit,
  System.Security.Permissions,
  Windows;

type
  TTypesArray = array of System.Type;
  TMethodsArray = array of MethodInfo;

  ClassImportAttribute = class(Attribute)
  private
    FUnitName, FClassName, FConstructorName, FDestructorName: string;
    FParameterTypes: TTypesArray;

  public
    constructor Create(AUnitName, AClassName, AConstructorName,
      ADestructorName: string); overload;
    constructor Create(AUnitName, AClassName, AConstructorName,
      ADestructorName: string; const AParameterTypes: TTypesArray); overload;

    function get_ParameterCount: Integer;

    property UnitName: string read FUnitName;
    property ClassName: string read FClassName;
    property ConstructorName: string read FConstructorName;
    property DestructorName: string read FDestructorName;
    property ParameterTypes: TTypesArray read FParameterTypes;
    property ParameterCount: Integer read get_ParameterCount;
  end;

  LibraryInterfaceAttribute = class(Attribute)
  public
    CallingConvention: System.Runtime.InteropServices.CallingConvention;
    CharSet: System.Runtime.InteropServices.CharSet;
    SetLastError: Boolean;

    constructor Create;
  end;

  LibraryMarshalAsAttribute = class(Attribute)
  public
    Value: UnmanagedType;
//    ArraySubType: UnmanagedType;
//    MarshalCookie: string;
//    MarshalType: string;
//    MarshalTypeRef: System.Type;
//    SafeArraySubType: VarEnum;
//    SafeArrayUserDefinedSubType: System.Type;
    SizeConst: Integer;
//    SizeParamIndex: short;

    constructor Create(AValue: UnmanagedType);
  end;

  function Supports(ModuleName: string; Source: System.Type; var Instance): Boolean; overload;

type
  IInstance = interface
    function GetModule: HModule;
    function GetModuleName: string;
  end;

  IInternalInstance = interface(IInstance)
    procedure SetModule(Value: HModule);
    procedure SetModuleName(Value: string);
    procedure SetOwner(Value: Boolean);
  end;

  [SecurityPermission(SecurityAction.LinkDemand, UnmanagedCode=True)]
  TLibraryInterface = class(TObject, IInstance, IInternalInstance, IDisposable)
  private
    FModule: HModule;
    FOwner: Boolean;
    FModuleName: string;

  strict protected
    procedure Finalize; override;

  public
    // IInstance
    function GetModule: HModule;
    function GetModuleName: string;

    // IInternalInstance
    procedure SetModule(Value: HModule);
    procedure SetModuleName(Value: string);
    procedure SetOwner(Value: Boolean);

    // IDisposable
    procedure Dispose; virtual;
  end;

implementation

uses
  SysUtils, System.Collections, System.Security;

var
  FDynamicTypes: ArrayList = nil;
  FDynamicClasses: ArrayList = nil;
  FClassInstances: HashTable = nil;
  FPackages: ArrayList = nil;

type
  [SecurityPermission(SecurityAction.LinkDemand, UnmanagedCode=True)]
  UnmanagedPackage = class
  public
    // Creates a wrapper and assigns the this pointer.
    class function CreateWrapper(ModuleName: string; Source: System.Type;
      var Instance; ThisPointer: IntPtr): Boolean; static;

    // Creates a wrapper.
    class function CreateInstance(ModuleName: string; Source: System.Type;
      var Instance): Boolean; overload; static;
    class function CreateInstance(ModuleName: string; Source: System.Type; var Instance;
      const Parameters: array of TObject): Boolean; overload; static;
    class function Equals(Obj1, Obj2: IInterface): Boolean;
    class procedure DestroyInstance(ThisPointer: IntPtr);
  end;

  IPackage = interface
    procedure Initialize;
    procedure Finalize;
  end;

  IWrapperClass = interface
    function GetThis: IntPtr;
  end;

  IInternalWrapperClass = interface(IWrapperClass)
    procedure SetThis(Value: IntPtr);
  end;

  [SecurityPermission(SecurityAction.LinkDemand, UnmanagedCode=True)]
  TWrapperClass = class(TLibraryInterface, IWrapperClass, IInternalWrapperClass)
  private
    FThis: IntPtr;

  strict protected
    procedure Finalize; override;

  public
    procedure Cleanup;

    // IWrapperClass
    function GetThis: IntPtr;

    // IInternalWrapperClass
    procedure SetThis(Value: IntPtr);
  end;

type
  TLibraryInterfaceItem = class
  private
    FModuleName: string;
    FProxyType, FInstanceType: System.Type;

  public
    constructor Create(AModuleName: string; AProxyType: System.Type);

    property ModuleName: string read FModuleName;
    property ProxyType: System.Type read FProxyType write FProxyType;
    property InstanceType: System.Type read FInstanceType write FInstanceType;
  end;

  TWrapperClassItem = class
  private
    FPackage: IPackage;
    FModuleName: string;
    FProxyType, FInstanceType: System.Type;
    FCreator: MethodInfo;

  public
    constructor Create(APackage: IPackage; AModuleName: string;
      AProxyType: System.Type);
    function get_Creator: MethodInfo;

    property Package: IPackage read FPackage;
    property ModuleName: string read FModuleName;
    property ProxyType: System.Type read FProxyType write FProxyType;
    property InstanceType: System.Type read FInstanceType write FInstanceType;
    property Creator: MethodInfo read get_Creator write FCreator;
  end;

  TPackageInstanceItem = class
  private
    FPackage: IPackage;
    FPackageName: string;

  public
    constructor Create(APackage: IPackage; APackageName: string);

    property Package: IPackage read FPackage;
    property PackageName: string read FPackageName;
  end;

  TRecurseInterfaces = class
  private
    FAllMethods: ArrayList;

  public
    procedure IterateInterfaces(Source: System.Type);
    function GetAllMethods(Source: System.Type): TMethodsArray;
  end;

{ TLibraryInterfaceItem }

constructor TLibraryInterfaceItem.Create(AModuleName: string; AProxyType: System.Type);
begin
  inherited Create;
  FModuleName := AModuleName;
  FProxyType := AProxyType;
  FInstanceType := nil;
end;

{ TNativeClassItem }

constructor TWrapperClassItem.Create(APackage: IPackage; AModuleName: string;
  AProxyType: System.Type);
begin
  inherited Create;
  FPackage := APackage;
  FModuleName := AModuleName;
  FProxyType := AProxyType;
  FInstanceType := nil;
  FCreator := nil;
end;

function TWrapperClassItem.get_Creator: MethodInfo;
begin
  Result := FCreator;
end;

function CreateDynamicModule: ModuleBuilder;
const
  SDynamicAssemblyName = 'DynamicAssembly'; // Do not localize.
  SDynamicModuleName = 'DynamicModule'; // Do not localize.
var
  DynamicAssembly: AssemblyBuilder;
  Name: AssemblyName;
begin
  Name := AssemblyName.Create;
  Name.Name := SDynamicAssemblyName + Guid.NewGuid().ToString('N');
  DynamicAssembly := AppDomain.CurrentDomain.DefineDynamicAssembly(Name,
    AssemblyBuilderAccess.RunAndSave);
  Result := DynamicAssembly.DefineDynamicModule(
    SDynamicModuleName + Guid.NewGuid().ToString('N'));
end;

function GetClassImportAttribute(AType: System.Type): ClassImportAttribute;
var
  Attributes: array of TObject;
begin
  Result := nil;
  Attributes := AType.GetCustomAttributes(typeof(ClassImportAttribute), False);

  if Length(Attributes) > 0 then
    Result := ClassImportAttribute(Attributes[0]);
end;

// This function takes a MethodInfo's parameter list and returns an array
// of all the types. If Translate is True and a parameter contains the
// ClassImportAttribute and the constructor takes no parameters then the
// type is modified from an interface to an IntPtr for the this pointer of the
// class.
function TypesOfParameters(const Parameters: array of ParameterInfo; Translate: Boolean = False): TTypesArray;
var
  Index, Size: Integer;
  ParameterType: System.Type;
  ClassAttribute: ClassImportAttribute;
begin
  Size := Length(Parameters);

  if Size = 0 then
    Result := nil
  else
  begin
    SetLength(Result, Size);

    for Index := 0 to Size - 1 do
    begin
      ParameterType := Parameters[Index].ParameterType;
      ClassAttribute := GetClassImportAttribute(ParameterType);

      if (Translate = True) and (ClassAttribute <> nil) and
        (ClassAttribute.ParameterCount = 0) then
      begin
        Result[Index] := typeof(IntPtr) // The self pointer of native class.
      end
      else
        Result[Index] := Parameters[Index].ParameterType;
    end;
  end;
end;

function UniqueTypeName(Prefix: string): string;
begin
  Result := Format('%s%s', [Prefix, Guid.NewGuid().ToString('N')]);
end;

// This function takes a MethodInfo's parameter list and returns an array
// of all the types with the this pointer as the first parameter. If Translate
// is True and a parameter contains the ClassImportAttribute and the
// constructor takes no parameters then the type is modified from an interface
// to an IntPtr for the this pointer of the class.
function TypesOfParametersWithThis(const Parameters: array of ParameterInfo;
  Translate: Boolean = False): TTypesArray;
var
  Index, Size: Integer;
  ParameterType: System.Type;
  ClassAttribute: ClassImportAttribute;
begin
  Size := Length(Parameters);
  SetLength(Result, Size + 1);
  Result[0] := typeof(IntPtr);

  if Size > 0 then
  begin
    for Index := 0 to Size - 1 do
    begin
      ParameterType := Parameters[Index].ParameterType;
      ClassAttribute := GetClassImportAttribute(ParameterType);

      if (Translate = True) and (ClassAttribute <> nil) and
        (ClassAttribute.ParameterCount = 0) then
      begin
        Result[Index + 1] := typeof(IntPtr) // The this pointer of native class.
      end
      else
        Result[Index + 1] := Parameters[Index].ParameterType;
    end;
  end;
end;

function FindType(ModuleName: string; Source: System.Type; var Item: TLibraryInterfaceItem): Boolean;
var
  Index: Integer;
  TempItem: TLibraryInterfaceItem;
begin
  Result := False;
  Item := nil;
  Monitor.Enter(FDynamicTypes);

  for Index := 0 to FDynamicTypes.Count - 1 do
  begin
    TempItem := TLibraryInterfaceItem(FDynamicTypes[Index]);

    if (TempItem <> nil) and (CompareStr(TempItem.ModuleName, ModuleName) <> 0) and
      (TempItem.ProxyType = Source) then
    begin
      Result := True;
      Item := TempItem;
    end;
  end;

  Monitor.Exit(FDynamicTypes);
end;

function FindClass(ModuleName: string; Source: System.Type; var Item: TWrapperClassItem): Boolean;
var
  Index: Integer;
  TempItem: TWrapperClassItem;
begin
  Result := False;
  Item := nil;
  Monitor.Enter(FDynamicClasses);

  for Index := 0 to FDynamicClasses.Count - 1 do
  begin
    TempItem := TWrapperClassItem(FDynamicClasses[Index]);

    if (TempItem <> nil) and (CompareStr(TempItem.ModuleName, ModuleName) <> 0) and
      (TempItem.ProxyType = Source) then
    begin
      Result := True;
      Item := TempItem;
    end;
  end;

  Monitor.Exit(FDynamicClasses);
end;

function MethodToMangledString(Method: MethodInfo): string;
var
  ParamInfo: array of ParameterInfo;
  Parameter: ParameterInfo;
  Index, Size: Integer;
  Attributes: array of TObject;
  Attribute: ClassImportAttribute;
  mangledname: string;
begin
  Result := '@' + Method.Name + '$qqs';
  ParamInfo := Method.GetParameters();
  Size := Length(ParamInfo);

  if Size = 0 then
    Result := Result + 'v'
  else
  begin
    for Index := 0 to Size - 1 do
    begin
      Parameter := ParamInfo[Index];
                                  
{
Integer	-2147483648..2147483647	signed 32-bit
Cardinal	0..4294967295	unsigned 32-bit
Fundamental integer types include Shortint, Smallint, Longint, Int64, Byte, Word, and Longword.

Fundamental integer types  
Type	Range	Format
Shortint	-128..127	signed 8-bit
Smallint	-32768..32767	signed 16-bit
Longint	-2147483648..2147483647	signed 32-bit
Int64	-2^63..2^63-1	signed 64-bit
Byte	0..255	unsigned 8-bit
Word	0..65535	unsigned 16-bit
Longword	0..4294967295	unsigned 32-bit

Real48	2.9 x 10^-39 .. 1.7 x 10^38	11-12	6
Single	1.5 x 10^-45 .. 3.4 x 10^38	7-8	4
Double	5.0 x 10^-324 .. 1.7 x 10^308	15-16	8
Extended	3.6 x 10^-4951 .. 1.1 x 10^4932	19-20	10
Comp	-2^63+1 .. 2^63 -1	19-20	8
Currency	-922337203685477.5808.. 922337203685477.5807	19-20	8
The generic type Real, in its current implementation, is equivalent to Double.

Generic real types 
Type	Range	Significant digits	Size in bytes
Real	5.0 x 10^-324 .. 1.7 x 10^308	15-16	8
}

      //Console.WriteLine(Parameter.ParameterType.ToString());
      if (parameter.ParameterType = typeof(char)) then
        Result := Result + 'c'
      else if (parameter.ParameterType = typeof(Shortint)) then
        Result := Result + 's'
      else if (parameter.ParameterType = typeof(Integer)) then
        Result := Result + 'i'
      else if (parameter.ParameterType = typeof(Longint)) then
        Result := Result + 'l'
      else if (parameter.ParameterType = typeof(Single)) then
        Result := Result + 'f'
      else if (parameter.ParameterType = typeof(Double)) then
        Result := Result + 'd'
      else if (parameter.ParameterType = typeof(string)) then
        Result := Result + 'pb' // On the Delph side a string is a PWideChar.
      else
      begin
        Attributes := Parameter.ParameterType.GetCustomAttributes(typeof(ClassImportAttribute), false);

        if Length(Attributes) > 0 then
        begin
          Attribute := ClassImportAttribute(attributes[0]);
          mangledname := attribute.UnitName + '@' + attribute.ClassName;
          Result := Result + 'p' + IntToStr(Length(mangledname)) + mangledname;
        end;

                                                                        
        // temporary.
        Console.WriteLine(parameter.ParameterType.ToString());
      end;
    end;
  end;
end;

// Only initialize each package once.
procedure InitializeOnce(Package: IPackage; PackageName: string);
var
  Index: Integer;
  PackageItem: TPackageInstanceItem;
  Found: Boolean;
begin
  Found := False;

  for Index := 0 to FPackages.Count - 1 do
  begin
    PackageItem := TPackageInstanceItem(FPackages[Index]);

    if CompareStr(PackageItem.PackageName, PackageName) <> 0 then
    begin
      Found := True;
      Break;
    end;
  end;

  if not Found then
  begin
    Package.Initialize();
    PackageItem := TPackageInstanceItem.Create(Package, PackageName);
    FPackages.Add(PackageItem);
  end;
end;

function CreateClass(Index: Integer; Instance: IInstance;
  const Parameters: array of TObject): IntPtr; overload;
var
  Item: TWrapperClassItem;
begin
  Result := IntPtr.Zero;

  if Index <> -1 then
  begin
    Monitor.Enter(FDynamicClasses);
    Item := TWrapperClassItem(FDynamicClasses[Index]);
    Monitor.Exit(FDynamicClasses);

    if Item <> nil then
    begin
      // Construct the object in the package.
      Result := IntPtr(Item.Creator.Invoke(nil, Parameters));
      IInternalWrapperClass(Instance).SetThis(Result);
    end;
  end;
end;

function CreateClass(Index: Integer; Instance: IInstance): IntPtr; overload;
begin
  Result := CreateClass(Index, Instance, []);
end;

function FindAttribute(const Attributes: array of TObject; AttributeType: System.Type): TObject;
var
  Index: Integer;
  A: TObject;
begin
  Result := nil;

  for Index := 0 to Length(Attributes) - 1 do
  begin
    A := Attributes[Index];

    if AttributeType = TypeOf(A) then
    begin
      Result := A;
      Break;
    end;
  end;
end;

function CopyAttribute(Attribute: Attribute): CustomAttributeBuilder; overload;
var
  AttributeConstructor: ConstructorInfo;
//  Parameters: array of System.Type;
  ParameterValues: array of TObject;
begin
  Result := nil;
  AttributeConstructor := nil;
  // Note: This is where we would try to determine the constructor used to
  // create the attribute and call it or call the default constructor. The
  // .NET 1.1 Framework does not support getting the constructor and parameter
  // values that a type was initialized with.

  if Attribute is MarshalAsAttribute then
  begin
    AttributeConstructor := TypeOf(Attribute).GetConstructor([TypeOf(UnmanagedType)]);
    SetLength(ParameterValues, 1);
    ParameterValues[0] := MarshalAsAttribute(Attribute).ArraySubType;
    Result := CustomAttributeBuilder.Create(AttributeConstructor,
      ParameterValues);
  end;
end;

function CopyAttribute(const Attributes: array of TObject; Index: Integer): CustomAttributeBuilder; overload;
begin
  Result := nil;

  if Attributes[Index] is Attribute then
    Result := CopyAttribute(Attributes[Index] as Attribute);
end;

procedure ApplyParameterAttributes(DestMethod: MethodBuilder;
  const Parameters: array of ParameterInfo); overload;
var
  Index: Integer;
//  DestFieldIndex, SourceFieldIndex: Integer;
  ParamAttributes: array of TObject;
  CustomAttribute: CustomAttributeBuilder;
  Parameter: ParameterBuilder;
  AttributeConstructor: ConstructorInfo;
//  MarshalAttribute: LibraryMarshalAsAttribute;
//  Temp: TObject;
//  FieldValues: array of TObject;
//  DestFields, SourceFields, AttributeFields: array of FieldInfo;
//  Properties: array of PropertyInfo;
begin
  for Index := 0 to Length(Parameters) - 1 do
  begin
    ParamAttributes := Parameters[Index].GetCustomAttributes(True);
    Parameter := DestMethod.DefineParameter(Index + 1,
      Parameters[Index].Attributes, Parameters[Index].Name);

    if Parameters[Index].IsIn then
    begin
      AttributeConstructor := TypeOf(InAttribute).GetConstructor([]);
      CustomAttribute := CustomAttributeBuilder.Create(AttributeConstructor, []);
      Parameter.SetCustomAttribute(CustomAttribute);
    end;

    if Parameters[Index].IsOut then
    begin
      AttributeConstructor := TypeOf(OutAttribute).GetConstructor([]);
      CustomAttribute := CustomAttributeBuilder.Create(AttributeConstructor, []);
      Parameter.SetCustomAttribute(CustomAttribute);
    end;
{
    Temp := FindAttribute(ParamAttributes, TypeOf(LibraryMarshalAsAttribute));

    if Temp <> nil then
    begin
      MarshalAttribute := LibraryMarshalAsAttribute(Temp);
      SourceFields := TypeOf(LibraryMarshalAsAttribute).GetFields;
      DestFields := TypeOf(MarshalAsAttribute).GetFields;

      for DestFieldIndex := 0 to Length(DestFields) - 1 do
      begin
        if CompareStr(DestFields[DestFieldIndex].Name, 'SizeConst') = 0 then // Do not localize.
        begin
          for SourceFieldIndex := 0 to Length(SourceFields) - 1 do
          begin
            if (CompareStr(SourceFields[SourceFieldIndex].Name,
              'SizeConst') = 0) then // Do not localize.
            begin
              try
                // This will throw an exception if it isn't a valid object.
                WriteLn(SourceFields[SourceFieldIndex].GetValue(MarshalAttribute).ToString);
                SetLength(AttributeFields, Length(AttributeFields) + 1);
                AttributeFields[Length(AttributeFields) - 1] := DestFields[DestFieldIndex];
                SetLength(FieldValues, Length(FieldValues) + 1);
                FieldValues[Length(FieldValues) - 1] := SourceFields[SourceFieldIndex].GetValue(MarshalAttribute);
              except
              end;
              Break;
            end;
          end;
          Break;
        end;
      end;

      SetLength(Properties, 0);
      AttributeConstructor := TypeOf(MarshalAsAttribute).GetConstructor([TypeOf(MarshalAttribute.Value)]);
      CustomAttribute := CustomAttributeBuilder.Create(AttributeConstructor,
        [MarshalAttribute.Value]);//, Properties,
//        [], AttributeFields, FieldValues);
      Parameter.SetCustomAttribute(CustomAttribute);
    end;
}
  end;
end;

procedure ApplyMethodAttributes(DestMethod: MethodBuilder; SourceMethod: MethodInfo);
var
//  Index: Integer;
  CustomAttribute: CustomAttributeBuilder;
//  Parameters: array of ParameterInfo;
  AttributeConstructor: ConstructorInfo;
begin
  if FindAttribute(SourceMethod.GetCustomAttributes(True),
    TypeOf(SuppressUnmanagedCodeSecurityAttribute)) <> nil then
  begin
    AttributeConstructor := TypeOf(SuppressUnmanagedCodeSecurityAttribute).GetConstructor([]);
    CustomAttribute := CustomAttributeBuilder.Create(AttributeConstructor, []);
    DestMethod.SetCustomAttribute(CustomAttribute);
  end;

  ApplyParameterAttributes(DestMethod, SourceMethod.GetParameters());

  // Note: It isn't currently possible to apply attributes to the return
  // type with MethodBuilder.
end;

function Supports(ModuleName: string; Source: System.Type; var Instance): Boolean;
const
  SProxyClassName = 'ProxyClass'; // Do not localize.
  SInstanceClass = 'InstanceClass'; // Do not localize.
var
  PInvokeMethod, Method: MethodBuilder;
  ProxyBuilder, InstanceBuilder: TypeBuilder;
  ParamIndex, Index, Size: Integer;
  Methods: array of MethodInfo;
  ParameterTypes: TTypesArray;
  MethodIL: ILGenerator;
  SourceMethod, InternalTypeMethod: MethodInfo;
  Module: HModule;
  Item: TLibraryInterfaceItem;
  RecurseInterfaces: TRecurseInterfaces;
  DynamicModule: ModuleBuilder;
//  MethodName: string;
  CallConv: CallingConvention;
  CharSet: System.Runtime.InteropServices.CharSet;
  Temp: TObject;
begin
  Result := False;
  Instance := nil;
  Module := Windows.LoadLibrary(ModuleName);

  if Module <> 0 then
  begin
    RecurseInterfaces := TRecurseInterfaces.Create;
    Methods := RecurseInterfaces.GetAllMethods(Source);
    Size := Length(Methods);

    if FindType(ModuleName, Source, Item) = False then
    begin
      Item := TLibraryInterfaceItem.Create(ModuleName, Source);
      Monitor.Enter(FDynamicTypes);
      FDynamicTypes.Add(Item);
      Monitor.Exit(FDynamicTypes);

      // Check to make sure all methods are in the module before doing anything.
      for Index := 0 to Size - 1 do
      begin
        SourceMethod := Methods[Index];

        if Windows.GetProcAddress(Module, SourceMethod.Name) = nil then
        begin
          Monitor.Enter(FDynamicTypes);
          FDynamicTypes.Remove(Item);
          Monitor.Exit(FDynamicTypes);
          Exit;
        end;
      end;

      DynamicModule := CreateDynamicModule;

      // Add the private proxy class to the module.
      ProxyBuilder := DynamicModule.DefineType(UniqueTypeName(SProxyClassName),
        TypeAttributes.NotPublic);

      for Index := 0 to Size - 1 do
      begin
        SourceMethod := Methods[Index];
        ParameterTypes := TypesOfParameters(SourceMethod.GetParameters, True);
        Temp := FindAttribute(SourceMethod.GetCustomAttributes(True),
          TypeOf(LibraryInterfaceAttribute));

        if Temp = nil then
        begin
          CallConv := CallingConvention.Winapi;
          CharSet := System.Runtime.InteropServices.CharSet.Auto;
        end
        else
        begin
          CallConv := LibraryInterfaceAttribute(Temp).CallingConvention;
          CharSet := LibraryInterfaceAttribute(Temp).CharSet;
        end;

        PInvokeMethod := ProxyBuilder.DefinePInvokeMethod(SourceMethod.Name,
          ModuleName,	MethodAttributes.Public or MethodAttributes.Static or
          MethodAttributes.PinvokeImpl or MethodAttributes.HideBySig,
          CallingConventions.Standard, Methods[Index].ReturnType{Return Type},
          ParameterTypes{Parameters}, CallConv,
          CharSet);

        // Apply preservesig metadata attribute so we can handle return
        // HRESULT ourselves.
        PInvokeMethod.SetImplementationFlags(MethodImplAttributes.Managed or
          MethodImplAttributes.PreserveSig or PInvokeMethod.GetMethodImplementationFlags());

        ApplyMethodAttributes(PInvokeMethod, SourceMethod);
      end;

      // Bake the type.
      Item.ProxyType := ProxyBuilder.CreateType();

      // Add a public class to the module.
      InstanceBuilder := DynamicModule.DefineType(UniqueTypeName(SInstanceClass),
        TypeAttributes.Public, typeof(TLibraryInterface), [Source]);

      // Implement the Source interface.
      InstanceBuilder.AddInterfaceImplementation(Source);

      for Index := 0 to Size - 1 do
      begin
        SourceMethod := Methods[Index];
        ParameterTypes := TypesOfParameters(SourceMethod.GetParameters);
        Method := InstanceBuilder.DefineMethod(SourceMethod.Name,
          MethodAttributes.Public or MethodAttributes.Virtual,
          Methods[Index].ReturnType{Return Type}, ParameterTypes{Parameter});

        // Generate IL for method.
        MethodIL := Method.GetILGenerator();
        MethodIL.Emit(OpCodes.Nop);

        if ParameterTypes <> nil then
        begin
          for ParamIndex := 0 to Length(ParameterTypes) - 1 do
          begin
            case ParamIndex of
              0: MethodIL.Emit(OpCodes.Ldarg_1);
              1: MethodIL.Emit(OpCodes.Ldarg_2);
              2: MethodIL.Emit(OpCodes.Ldarg_3);
              else
                MethodIL.Emit(OpCodes.Ldarg_S, ParamIndex + 1);
            end;
          end;
        end;

        InternalTypeMethod := Item.ProxyType.GetMethod(SourceMethod.Name);
        MethodIL.Emit(OpCodes.Call, InternalTypeMethod);
        MethodIL.Emit(OpCodes.Nop);
        MethodIL.Emit(OpCodes.Ret);
      end;

      // Bake the type.
      Item.InstanceType := InstanceBuilder.CreateType();
    end;

    Instance := Activator.CreateInstance(Item.InstanceType) as IInterface;
    Result := True;
  end;
end;

function GenerateWrapper(ModuleName: string; Source: System.Type;
  var Instance): Integer;
const
  SProxyClassName = 'ProxyClass'; // Do not localize.
  SInstanceClass = 'InstanceClass'; // Do not localize.
  SGetThis = 'GetThis'; // Do not localize.
  SGetModuleName = 'GetModuleName'; // Do not localize.
  SGetTypeFromHandle = 'GetTypeFromHandle'; // Do not localize.
  SCreateWrapper = 'CreateWrapper'; // Do not localize.
  SDispose = 'Dispose'; // Do not localize.
  SCleanup = 'Cleanup'; // Do not localize.
var
  Package: IPackage;
  Dll: HModule;
  Method: MethodBuilder;
  ProxyBuilder, InstanceBuilder: TypeBuilder;
  ReturnType: System.Type;
  ParamIndex, Index, Size: Integer;
  CurrentMethod: MethodInfo;
  MethodIL: ILGenerator;
  ClassAttribute, MethodAttribute: ClassImportAttribute;
  Methods: array of MethodInfo;
  MangledNames: array of string;
  ParameterTypes: TTypesArray;
  MangledName: string;
  RecurseInterfaces: TRecurseInterfaces;
  DynamicModule: ModuleBuilder;
  Item: TWrapperClassItem;
  TempResult: Integer;
begin
  Result := -1;
  Instance := nil;
  ClassAttribute := GetClassImportAttribute(Source);

  if (ClassAttribute <> nil) and
    (Supports(ModuleName, typeof(IPackage), Package) = True) then
  begin
    Item := nil;
    TempResult := -1;
    Dll := Windows.LoadLibrary(ModuleName);

    if FindClass(ModuleName, Source, Item) = False then
    begin
      Item := TWrapperClassItem.Create(Package, ModuleName, Source);
      Monitor.Enter(FDynamicClasses);
      TempResult := FDynamicClasses.Add(Item);
      Monitor.Exit(FDynamicClasses);
      RecurseInterfaces := TRecurseInterfaces.Create;
      Methods := RecurseInterfaces.GetAllMethods(Source);
      RecurseInterfaces := nil;
      Size := Length(Methods);

      if Dll <> 0 then
      begin
        SetLength(MangledNames, Size);

        if Windows.GetProcAddress(Dll, ClassAttribute.ConstructorName) = nil then
        begin
          Monitor.Enter(FDynamicClasses);
          FDynamicClasses.Remove(Item);
          Monitor.Exit(FDynamicClasses);
          Exit;
        end;

        if Windows.GetProcAddress(Dll, ClassAttribute.DestructorName) = nil then
        begin
          Monitor.Enter(FDynamicClasses);
          FDynamicClasses.Remove(Item);
          Monitor.Exit(FDynamicClasses);
          Exit;
        end;

        // Check to make sure all methods are in the module before we do anything.
        for Index := 0 to Size - 1 do
        begin
          MangledName := Format('@%s@%s%s', [ClassAttribute.UnitName,
            ClassAttribute.ClassName, MethodToMangledString(Methods[index])]);

          if Windows.GetProcAddress(Dll, MangledName) = nil then
          begin
            Monitor.Enter(FDynamicClasses);
            FDynamicClasses.Remove(Item);
            Monitor.Exit(FDynamicClasses);
            Exit;
          end;

          MangledNames[index] := MangledName;
        end;

        DynamicModule := CreateDynamicModule;

        // Add module to assembly.
        ProxyBuilder := DynamicModule.DefineType(UniqueTypeName(SProxyClassName + ClassAttribute.ClassName),
          TypeAttributes.NotPublic);

        // Add constructor and destructor to the proxy type.
        Method := ProxyBuilder.DefinePInvokeMethod(ClassAttribute.ConstructorName,
          ModuleName,	MethodAttributes.Public or MethodAttributes.Static or
          MethodAttributes.PinvokeImpl or MethodAttributes.HideBySig,
          CallingConventions.Standard, typeof(IntPtr){Return Type},
          ClassAttribute.ParameterTypes{Parameters}, CallingConvention.StdCall,
          CharSet.Auto);

        Method.SetImplementationFlags(MethodImplAttributes.Managed or
          MethodImplAttributes.PreserveSig or Method.GetMethodImplementationFlags());

        Method := ProxyBuilder.DefinePInvokeMethod(ClassAttribute.DestructorName,
          ModuleName,	MethodAttributes.Public or MethodAttributes.Static or
          MethodAttributes.PinvokeImpl or MethodAttributes.HideBySig,
          CallingConventions.Standard, nil{Return Type},
          [typeof(IntPtr)]{Parameters}, CallingConvention.StdCall,
          CharSet.Auto);

        Method.SetImplementationFlags(MethodImplAttributes.Managed or
          MethodImplAttributes.PreserveSig or Method.GetMethodImplementationFlags());

        for Index := 0 to Size - 1 do
        begin
          ParameterTypes := TypesOfParametersWithThis(Methods[index].GetParameters(), True);
          MethodAttribute := GetClassImportAttribute(Methods[Index].ReturnType);

          if MethodAttribute <> nil then
            ReturnType := typeof(IntPtr)
          else
            ReturnType := Methods[Index].ReturnType;

          Method := ProxyBuilder.DefinePInvokeMethod(MangledNames[index],
            ModuleName,	MethodAttributes.Public or MethodAttributes.Static or
            MethodAttributes.PinvokeImpl or MethodAttributes.HideBySig,
            CallingConventions.Standard, ReturnType{Return Type},
            ParameterTypes{Parameters}, CallingConvention.StdCall,
            CharSet.Auto);

          // Apply preservesig metadata attribute so we can handle return
          // HRESULT ourselves.
          Method.SetImplementationFlags(MethodImplAttributes.Managed or
            MethodImplAttributes.PreserveSig or Method.GetMethodImplementationFlags());
        end;

        // Bake the type.
        Item.ProxyType := ProxyBuilder.CreateType();

        // Add a public class to the module.
        InstanceBuilder := DynamicModule.DefineType(UniqueTypeName(SInstanceClass + ClassAttribute.ClassName),
          TypeAttributes.Public, typeof(TWrapperClass), [Source, typeof(IDisposable)]);

        // Implement the Source interface.
        InstanceBuilder.AddInterfaceImplementation(Source);

        // Implement IDisposable.Dispose.
        Method := InstanceBuilder.DefineMethod(SDispose,
          MethodAttributes.Public or MethodAttributes.Virtual,
          nil{Return Type}, nil{Parameters});

        MethodIL := Method.GetILGenerator();
        MethodIL.Emit(OpCodes.Ldarg_0);
        CurrentMethod := typeof(IWrapperClass).GetMethod(SGetThis);
        MethodIL.Emit(OpCodes.Callvirt, CurrentMethod);
        CurrentMethod := nil;
        CurrentMethod := Item.ProxyType.GetMethod(ClassAttribute.DestructorName);
        MethodIL.Emit(OpCodes.Call, CurrentMethod);
        CurrentMethod := nil;
        MethodIL.Emit(OpCodes.Ldarg_0);
        CurrentMethod := typeof(TWrapperClass).GetMethod(SCleanup);
        MethodIL.Emit(OpCodes.Callvirt, CurrentMethod);
        CurrentMethod := nil;
        MethodIL.Emit(OpCodes.Ret);

        // Implement the wrapper method for each method.
        for Index := 0 to Size - 1 do
        begin
          ParameterTypes := TypesOfParameters(Methods[Index].GetParameters());
          ReturnType := Methods[Index].ReturnType;
          MethodAttribute := GetClassImportAttribute(ReturnType);
          Method := InstanceBuilder.DefineMethod(Methods[Index].Name,
            MethodAttributes.Public or MethodAttributes.Virtual,
            ReturnType{Return Type}, ParameterTypes{Parameters});

          // Generate IL for method.
          MethodIL := Method.GetILGenerator();
          MethodIL.Emit(OpCodes.Ldarg_0);
          CurrentMethod := typeof(IWrapperClass).GetMethod(SGetThis);
          MethodIl.Emit(OpCodes.Callvirt, CurrentMethod);
          CurrentMethod := nil;

          if MethodAttribute <> nil then
          begin
            MethodIL.DeclareLocal(ReturnType);
            MethodIL.DeclareLocal(typeof(IntPtr));
            MethodIL.DeclareLocal(typeof(TObject));
            MethodIL.DeclareLocal(typeof(Integer));
          end;

          if ParameterTypes <> nil then
          begin
            for ParamIndex := 0 to Length(ParameterTypes) - 1 do
            begin
              case ParamIndex of
                0: MethodIL.Emit(OpCodes.Ldarg_1);
                1: MethodIL.Emit(OpCodes.Ldarg_2);
                2: MethodIL.Emit(OpCodes.Ldarg_3);
                else
                begin
                  MethodIL.Emit(OpCodes.Ldarg_S, Index + 1);
                end;
              end;

              // In cases where the parameter type is a native class interface
              // the parameter value is replaced with the this pointer of the
              // underlying native class.
              if MethodAttribute <> nil then
              begin
                MethodIL.Emit(OpCodes.Isinst, typeof(IWrapperClass));
                CurrentMethod := typeof(IWrapperClass).GetMethod(SGetThis);
                MethodIL.Emit(OpCodes.Callvirt, CurrentMethod);
                CurrentMethod := nil;
              end;
            end;
          end;

          CurrentMethod := Item.ProxyType.GetMethod(MangledNames[Index]);
          MethodIL.Emit(OpCodes.Call, CurrentMethod);
          CurrentMethod := nil;

          if MethodAttribute <> nil then
          begin
            // Store the this pointer so it can be assigned to the wrapper class.
            MethodIL.Emit(OpCodes.Stloc_1);

            // Load ModuleName parameter.
            MethodIL.Emit(OpCodes.Ldarg_0);
            MethodIL.Emit(OpCodes.Isinst, typeof(IInstance));
            CurrentMethod := typeof(IInstance).GetMethod(SGetModuleName);
            MethodIL.Emit(OpCodes.Callvirt, CurrentMethod);
            CurrentMethod := nil;

            // Load the Source parameter.
            MethodIL.Emit(OpCodes.Ldtoken, ReturnType);
            CurrentMethod := typeof(System.Type).GetMethod(SGetTypeFromHandle);
            MethodIL.Emit(OpCodes.Call, CurrentMethod);
            CurrentMethod := nil;

            // Load the Instance parameter.
            MethodIL.Emit(OpCodes.Ldloc_0);
            MethodIL.Emit(OpCodes.Isinst, typeof(TObject));
            MethodIL.Emit(OpCodes.Stloc_2);
            MethodIL.Emit(OpCodes.Ldloca_s, 2);

            // Load the this pointer for the returned type.
            MethodIL.Emit(OpCodes.Ldloc_1);

            // Call GenerateWrapper. Load the Index parameter.
            CurrentMethod := typeof(UnmanagedPackage).GetMethod(SCreateWrapper);
            MethodIL.Emit(OpCodes.Call, CurrentMethod);
            CurrentMethod := nil;
            MethodIL.Emit(OpCodes.Stloc_3);

            // Convert to the return type.
            MethodIL.Emit(OpCodes.Ldloc_2);
            MethodIL.Emit(OpCodes.Isinst, ReturnType);
            MethodIL.Emit(OpCodes.Stloc_0);

            // Load the return value.
            MethodIL.Emit(OpCodes.Ldloc_0);
          end;

          MethodIL.Emit(OpCodes.Ret);
        end;

        // Bake the type.
        Item.InstanceType := InstanceBuilder.CreateType();
        InitializeOnce(Package, ModuleName);
        Item.Creator := Item.ProxyType.GetMethod(ClassAttribute.ConstructorName);
      end;
    end;

    if Item <> nil then
    begin
      Instance := Activator.CreateInstance(Item.InstanceType);
      IInternalInstance(Instance).SetOwner(True);
      IInternalInstance(Instance).SetModule(DLL);
      IInternalInstance(Instance).SetModuleName(ModuleName);
      Result := TempResult;
    end;
  end;
end;

class function UnmanagedPackage.CreateWrapper(ModuleName: string; Source: System.Type;
  var Instance; ThisPointer: IntPtr): Boolean;
var
  Item: TObject;
begin
  Result := False;
  Item := FClassInstances[ThisPointer];

  if Item <> nil then
  begin
    Instance := Item;
  end
  else if GenerateWrapper(ModuleName, Source, Instance) <> 0 then
  begin
    IInternalWrapperClass(Instance).SetThis(ThisPointer);
    FClassInstances.Add(ThisPointer, IWrapperClass(Instance));
    Result := True;
  end;
end;

class function UnmanagedPackage.CreateInstance(ModuleName: string;
  Source: System.Type; var Instance): Boolean;
begin
  Result := CreateInstance(ModuleName, Source, Instance, []);
end;

class function UnmanagedPackage.CreateInstance(ModuleName: string;
  Source: System.Type; var Instance;
  const Parameters: array of TObject): Boolean;
var
  Index: Integer;
begin
  Index := GenerateWrapper(ModuleName, Source, Instance);
  Result := CreateClass(Index, IInstance(Instance), Parameters) <> IntPtr.Zero;
end;

// Question: If Obj1 and Obj2 do not implement the interface IWrapperClass should an
// exception be raised?
class function UnmanagedPackage.Equals(Obj1, Obj2: IInterface): Boolean;
var
  Inst1, Inst2: IWrapperClass;
begin
  if SysUtils.Supports(Obj1, IWrapperClass, Inst1) and
    SysUtils.Supports(Obj2, IWrapperClass, Inst2) and
    (Inst1.GetThis = Inst2.GetThis) then
  begin
    Result := True;
  end
  else
    Result := False;
end;

class procedure UnmanagedPackage.DestroyInstance(ThisPointer: IntPtr);
begin
  if FClassInstances.Contains(ThisPointer) then
    FClassInstances.Remove(ThisPointer);
end;

{ ClassImportAttribute }

constructor ClassImportAttribute.Create(AUnitName, AClassName,
  AConstructorName, ADestructorName: string);
begin
  inherited Create;
  FUnitName := AUnitName;
  FClassName := AClassName;
  FConstructorName := AConstructorName;
  FDestructorName := ADestructorName;
  FParameterTypes := nil;
end;

constructor ClassImportAttribute.Create(AUnitName, AClassName, AConstructorName,
  ADestructorName: string; const AParameterTypes: TTypesArray);
begin
  inherited Create;
  FUnitName := AUnitName;
  FClassName := AClassName;
  FConstructorName := AConstructorName;
  FDestructorName := ADestructorName;
  FParameterTypes := AParameterTypes;
end;

function ClassImportAttribute.get_ParameterCount: Integer;
begin
  Result := 0;

  if FParameterTypes <> nil then
    Result := Length(FParameterTypes);
end;

{ LibraryInterfaceAttribute }

constructor LibraryInterfaceAttribute.Create;
begin
  inherited Create;
  CallingConvention := System.Runtime.InteropServices.CallingConvention.Winapi;
  CharSet := System.Runtime.InteropServices.CharSet.Auto;
  SetLastError := False;
end;

{ LibraryMarshalAsAttribute }

constructor LibraryMarshalAsAttribute.Create(AValue: UnmanagedType);
begin
  inherited Create;
  Value := AValue;
  SizeConst := 0;
end;

{ TLibraryInterface }

procedure TLibraryInterface.Finalize;
begin
  inherited;
end;

function TLibraryInterface.GetModule: HModule;
begin
  Result := FModule;
end;

function TLibraryInterface.GetModuleName: string;
begin
  Result := FModuleName;
end;

procedure TLibraryInterface.SetModule(Value: HModule);
begin
  FModule := Value;
end;

procedure TLibraryInterface.SetModuleName(Value: string);
begin
  FModuleName := Value;
end;

procedure TLibraryInterface.SetOwner(Value: Boolean);
begin
  FOwner := Value;
end;

procedure TLibraryInterface.Dispose;
begin
  GC.SuppressFinalize(Self);

  if (FOwner = True) and (FModule <> 0) then
    FreeLibrary(FModule);
end;

{ TWrapperClass }

procedure TWrapperClass.Finalize;
begin
  inherited;

  if FThis <> IntPtr.Zero then
    Cleanup;
end;

procedure TWrapperClass.Cleanup;
begin
  UnmanagedPackage.DestroyInstance(FThis);
  FThis := IntPtr.Zero;
end;

function TWrapperClass.GetThis: IntPtr;
begin
  Result := FThis;
end;

procedure TWrapperClass.SetThis(Value: IntPtr);
begin
  FThis := Value;
end;

{ TRecurseInterfaces }

procedure TRecurseInterfaces.IterateInterfaces(Source: System.Type);
var
  Methods: array of MethodInfo;
  Index: Integer;
  Interfaces: array of System.Type;
begin
  Methods := Source.GetMethods();

  for Index := 0 to Length(Methods) - 1 do
    FAllMethods.Add(Methods[Index]);

  Interfaces := Source.GetInterfaces();

  for Index := 0 to Length(Interfaces) - 1 do
    IterateInterfaces(Interfaces[Index]);
end;

function TRecurseInterfaces.GetAllMethods(Source: System.Type): TMethodsArray;
var
  Index, Size: Integer;
begin
  // Note: FAllMethods stores the variable for the life of this method call
  // It needs to be cleared after this function is done.
  Result := nil;

  FAllMethods := ArrayList.Create();
  IterateInterfaces(Source);
  Size := FAllMethods.Count;
  SetLength(Result, Size);

  for Index := 0 to Size - 1 do
    Result[Index] := FAllMethods[Index] as MethodInfo;

  FAllMethods := nil;
end;

{ TPackageInstanceItem }

constructor TPackageInstanceItem.Create(APackage: IPackage; APackageName: string);
begin
  inherited Create;
  FPackage := APackage;
  FPackageName := APackageName;
end;

initialization
  FDynamicTypes := ArrayList.Create;
  FDynamicClasses := ArrayList.Create;
  FClassInstances := HashTable.Create;
  FPackages := ArrayList.Create;

end.
